<?php
// public/china_step1_submit.php
session_start();
require_once __DIR__ . '/../config/db.php';

// CSRF check
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== ($_SESSION['csrf_token'] ?? '')) {
    die('Invalid CSRF token');
}

$errors = [];
$old    = [];

// فیلدهایی که انتظار داریم
$fields = [
    'family_name','given_name','middle_name',
    'gender','dob','nationality','birth_country',
    'marital_status','phone','email'
];

// خواندن داده‌ها
foreach ($fields as $f) {
    $old[$f] = trim($_POST[$f] ?? '');
}

// Validation
// Family Name
if ($old['family_name'] === '' || !preg_match('/^[A-Za-z\s]{2,}$/', $old['family_name'])) {
    $errors['family_name'] = 'نام خانوادگی الزامی است و باید با حروف لاتین وارد شود.';
}

// Given Name
if ($old['given_name'] === '' || !preg_match('/^[A-Za-z\s]{2,}$/', $old['given_name'])) {
    $errors['given_name'] = 'نام الزامی است و باید با حروف لاتین وارد شود.';
}

// Gender
if (!in_array($old['gender'], ['male','female'], true)) {
    $errors['gender'] = 'لطفاً جنسیت را انتخاب کنید.';
}

// DOB
$dob = DateTime::createFromFormat('Y-m-d', $old['dob']);
$dobErrors = DateTime::getLastErrors();
if ($dob === false || $dobErrors['warning_count'] > 0 || $dobErrors['error_count'] > 0) {
    $errors['dob'] = 'تاریخ تولد نامعتبر است.';
} else {
    $now = new DateTime();
    $age = $now->diff($dob)->y;
    if ($age < 18) {
        $errors['dob'] = 'سن متقاضی باید حداقل ۱۸ سال باشد.';
    }
}

// Nationality
if ($old['nationality'] === '') {
    $errors['nationality'] = 'تابعیت الزامی است.';
}

// Birth Country
if ($old['birth_country'] === '') {
    $errors['birth_country'] = 'کشور محل تولد الزامی است.';
}

// Marital Status
if (!in_array($old['marital_status'], ['single','married','divorced','widowed'], true)) {
    $errors['marital_status'] = 'لطفاً وضعیت تأهل را انتخاب کنید.';
}

// Phone
if ($old['phone'] === '' || !preg_match('/^[0-9+\-\s]{8,20}$/', $old['phone'])) {
    $errors['phone'] = 'لطفاً شماره موبایل معتبر وارد کنید.';
}

// Email
if (!filter_var($old['email'], FILTER_VALIDATE_EMAIL)) {
    $errors['email'] = 'ایمیل نامعتبر است.';
}

// اگر خطا وجود دارد -> برگشت به فرم
if ($errors) {
    $_SESSION['form_errors'] = $errors;
    $_SESSION['form_old']    = $old;
    header('Location: china_step1.php');
    exit;
}

// اگر اینجا رسیدیم یعنی داده‌ها معتبرند
$pdo = get_pdo();
$pdo->beginTransaction();

// چک می‌کنیم آیا قبلاً application برای این سشن ساخته شده یا نه
$applicationId = $_SESSION['china_application_id'] ?? null;

try {
    if (!$applicationId) {
        // ایجاد رکورد جدید در visa_applications
        $stmt = $pdo->prepare("
            INSERT INTO visa_applications (country_code, visa_type, current_step, status, created_at, updated_at)
            VALUES ('CN', NULL, 1, 'draft', NOW(), NOW())
        ");
        $stmt->execute();
        $applicationId = $pdo->lastInsertId();
        $_SESSION['china_application_id'] = $applicationId;
    }

    // چک وجود applicant
    $stmt = $pdo->prepare("SELECT id FROM visa_applicants WHERE application_id = ?");
    $stmt->execute([$applicationId]);
    $existingId = $stmt->fetchColumn();

    if ($existingId) {
        $stmt = $pdo->prepare("
            UPDATE visa_applicants
            SET family_name = ?, given_name = ?, middle_name = ?, gender = ?, dob = ?, 
                nationality = ?, birth_country = ?, marital_status = ?, phone = ?, email = ?, updated_at = NOW()
            WHERE application_id = ?
        ");
        $stmt->execute([
            $old['family_name'],
            $old['given_name'],
            $old['middle_name'] ?: null,
            $old['gender'],
            $old['dob'],
            $old['nationality'],
            $old['birth_country'],
            $old['marital_status'],
            $old['phone'],
            $old['email'],
            $applicationId
        ]);
    } else {
        $stmt = $pdo->prepare("
            INSERT INTO visa_applicants
                (application_id, family_name, given_name, middle_name, gender, dob,
                 nationality, birth_country, marital_status, phone, email,
                 created_at, updated_at)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,NOW(),NOW())
        ");
        $stmt->execute([
            $applicationId,
            $old['family_name'],
            $old['given_name'],
            $old['middle_name'] ?: null,
            $old['gender'],
            $old['dob'],
            $old['nationality'],
            $old['birth_country'],
            $old['marital_status'],
            $old['phone'],
            $old['email'],
        ]);
    }

    $pdo->commit();
} catch (Exception $e) {
    $pdo->rollBack();
    // برای تست سریع:
    die('Database error: ' . $e->getMessage());
}

// TODO: بعداً این صفحه را می‌سازیم
header('Location: china_step2.php');
exit;
